/**************************************************************************************
 
   Copyright (c) Hilscher GmbH. All Rights Reserved.
 
 **************************************************************************************
 
   Filename:
    $Workfile: ToolkitSample.c $
   Last Modification:
    $Author: stephans $
    $Modtime: $
    $Revision: 6505 $
   
   Targets:
     rcX   : yes
 
   Description:
       
   Changes:
 
     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     2         23.07.2014  RM       Updated SerDPMIfWrite/SerDPMRead function parameters
                                    according to cifXToolkit (PFN_HWIF_MEMCPY) parameters
     1         03.10.2011  SS       Initial version derived from windows example
     
**************************************************************************************/

/* rcX includes */
#include <rX_Includes.h>
#include "AP_LED.h"
#include "UartStdIOHook/rX_UartStdIOHook.h"

/* Standard includes */
#include <stdio.h>
#include <unistd.h>

/* Toolkit includes */
#include "cifXToolkit.h"
#include "CIFXErrors.h"
#include "SerialDPM/SerialDPMInterface.h"
#include "SerialDPM/OS_Spi.h"
#include "ToolkitSample.h"

typedef struct PERFORMANCE_TEST_DATAtag {
  uint32_t ulMax;
  uint32_t ulMin;
  uint32_t ulSum;
  uint32_t ulCnt;

} PERFORMANCE_TEST_DATA, *PPERFORMANCE_TEST_DATA;

/*****************************************************************************/
/*! Thread for cyclic Toolkit timer, which handles polling of COS bits on
 *   non-irq cards
 *     \param arg      Pollinterval in ms
 *     \return NULL on termination                                            */
/*****************************************************************************/
static void cifXPollingThread(void* pvParam)
{
  uint32_t ulPollInterval = (uint32_t)pvParam;
   
  while (1)
  {
    cifXTKitCyclicTimer();
    OS_Sleep(ulPollInterval);
  }
}

/*****************************************************************************/
/*! Displays a hex dump on the debug console (16 bytes per line)
*   \param pbData     Pointer to dump data
*   \param ulDataLen  Length of data dump                                    */
/*****************************************************************************/
static void DumpData(uint8_t* pbData, uint32_t ulDataLen)
{
  uint32_t ulIdx = 0;
  for(ulIdx = 0; ulIdx < ulDataLen; ++ulIdx)
  {
    if(0 == (ulIdx % 16))
      printf("\r\n");

    printf("%02X ", pbData[ulIdx]);
  }
  printf("\r\n");
}

/*****************************************************************************/
/*! Dumps a rcX packet to debug console
*   \param ptPacket Pointer to packed being dumped                           */
/*****************************************************************************/
static void DumpPacket(CIFX_PACKET* ptPacket)
{
  printf("Dest   : 0x%08lX      ID   : 0x%08lX\r\n", ptPacket->tHeader.ulDest,   ptPacket->tHeader.ulId);
  printf("Src    : 0x%08lX      Sta  : 0x%08lX\r\n", ptPacket->tHeader.ulSrc,    ptPacket->tHeader.ulState);
  printf("DestID : 0x%08lX      Cmd  : 0x%08lX\r\n", ptPacket->tHeader.ulDestId, ptPacket->tHeader.ulCmd);
  printf("SrcID  : 0x%08lX      Ext  : 0x%08lX\r\n", ptPacket->tHeader.ulSrcId,  ptPacket->tHeader.ulExt);
  printf("Len    : 0x%08lX      Rout : 0x%08lX\r\n", ptPacket->tHeader.ulLen,    ptPacket->tHeader.ulRout);

  printf("Data:");
  DumpData(ptPacket->abData, ptPacket->tHeader.ulLen);
}

/*****************************************************************************/
/*! Function to demonstrate the board/channel enumeration
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
static int32_t EnumBoardDemo(void)
{
  CIFXHANDLE hDriver = NULL;
  int32_t lRet    = xDriverOpen(&hDriver);

  printf("\r\n---------- Board/Channel enumeration demo ----------\r\n");

  if(CIFX_NO_ERROR == lRet)
  {
    /* Driver/Toolkit successfully opened */
    uint32_t          ulBoard    = 0;
    BOARD_INFORMATION tBoardInfo = {0};

    /* Iterate over all boards */
    while(CIFX_NO_ERROR == xDriverEnumBoards(hDriver, ulBoard, sizeof(tBoardInfo), &tBoardInfo))
    {
      printf("Found Board %s\r\n", tBoardInfo.abBoardName);
      if(strlen( (char*)tBoardInfo.abBoardAlias) != 0)
        printf(" Alias        : %s\r\n", tBoardInfo.abBoardAlias);

      printf(" DeviceNumber : %lu\r\n", tBoardInfo.tSystemInfo.ulDeviceNumber);
      printf(" SerialNumber : %lu\r\n", tBoardInfo.tSystemInfo.ulSerialNumber);
      printf(" Board ID     : %lu\r\n", tBoardInfo.ulBoardID);
      printf(" System Error : 0x%08lX\r\n", tBoardInfo.ulSystemError);
      printf(" Channels     : %lu\r\n", tBoardInfo.ulChannelCnt);
      printf(" DPM Size     : %lu\r\n", tBoardInfo.ulDpmTotalSize);

      uint32_t            ulChannel    = 0;
      CHANNEL_INFORMATION tChannelInfo = {{0}};

      /* iterate over all channels on the current board */
      while(CIFX_NO_ERROR == xDriverEnumChannels(hDriver, ulBoard, ulChannel, sizeof(tChannelInfo), &tChannelInfo))
      {
        printf(" - Channel %lu:\r\n", ulChannel);
        printf("    Firmware : %s\r\n", tChannelInfo.abFWName);
        printf("    Version  : %u.%u.%u build %u\r\n", 
               tChannelInfo.usFWMajor,
               tChannelInfo.usFWMinor,
               tChannelInfo.usFWRevision,
               tChannelInfo.usFWBuild);
        printf("    Date     : %02u/%02u/%04u\r\n", 
               tChannelInfo.bFWMonth,
               tChannelInfo.bFWDay,
               tChannelInfo.usFWYear);

        ++ulChannel;
      }

      ++ulBoard;
    }

    /* close previously opened driver */
    xDriverClose(hDriver);
  }

  printf(" State = 0x%08lX\r\n", lRet);
  printf("----------------------------------------------------\r\n");

  return lRet;
}

/*****************************************************************************/
/*! Function to demonstrate system channel functionality (PacketTransfer)
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
static int32_t SysdeviceDemo()
{
  CIFXHANDLE hDriver = NULL;
  int32_t    lRet    = xDriverOpen(&hDriver);

  printf("\r\n---------- System Device handling demo ----------\r\n");

  if(CIFX_NO_ERROR == lRet)
  {
    /* Driver/Toolkit successfully opened */
    CIFXHANDLE hSys = NULL;
    lRet = xSysdeviceOpen(hDriver, "cifX0", &hSys);

    if(CIFX_NO_ERROR != lRet)
    {
      printf("Error opening SystemDevice!\r\n");

    } else
    {
      SYSTEM_CHANNEL_SYSTEM_INFO_BLOCK tSysInfo = {{0}};
     
      /* System channel successfully opened, try to read the System Info Block */
      if( CIFX_NO_ERROR != (lRet = xSysdeviceInfo(hSys, CIFX_INFO_CMD_SYSTEM_INFO_BLOCK, sizeof(tSysInfo), &tSysInfo)))
      {
        printf("Error querying system information block\r\n");
      } else
      {
        printf("System Channel Info Block:\r\n");
        printf("DPM Size         : %lu\r\n", tSysInfo.ulDpmTotalSize);
        printf("Device Number    : %lu\r\n", tSysInfo.ulDeviceNumber);
        printf("Serial Number    : %lu\r\n", tSysInfo.ulSerialNumber);
        printf("Manufacturer     : %u\r\n", tSysInfo.usManufacturer);
        printf("Production Date  : %u\r\n", tSysInfo.usProductionDate);
        printf("Device Class     : %u\r\n", tSysInfo.usDeviceClass);
        printf("HW Revision      : %u\r\n", tSysInfo.bHwRevision);
        printf("HW Compatibility : %u\r\n", tSysInfo.bHwCompatibility);
      }

      uint32_t ulSendPktCount = 0;
      uint32_t ulRecvPktCount = 0;

      /* Do a simple Packet exchange via system channel */
      xSysdeviceGetMBXState(hSys, &ulRecvPktCount, &ulSendPktCount);
      printf("System Mailbox State: MaxSend = %lu, Pending Receive = %lu\r\n",
             ulSendPktCount, ulRecvPktCount);

      CIFX_PACKET tSendPkt = {{0}};
      CIFX_PACKET tRecvPkt = {{0}};

      if(CIFX_NO_ERROR != (lRet = xSysdevicePutPacket(hSys, &tSendPkt, 10)))
      {
        printf("Error sending packet to device!\r\n");
      } else
      {
        printf("Send Packet:\r\n");
        DumpPacket(&tSendPkt);

        xSysdeviceGetMBXState(hSys, &ulRecvPktCount, &ulSendPktCount);
        printf("System Mailbox State: MaxSend = %lu, Pending Receive = %lu\r\n",
              ulSendPktCount, ulRecvPktCount);

        if(CIFX_NO_ERROR != (lRet = xSysdeviceGetPacket(hSys, sizeof(tRecvPkt), &tRecvPkt, 20)) )
        {
          printf("Error getting packet from device!\r\n");
        } else
        {
          printf("Received Packet:\r\n");
          DumpPacket(&tRecvPkt);

          xSysdeviceGetMBXState(hSys, &ulRecvPktCount, &ulSendPktCount);
          printf("System Mailbox State: MaxSend = %lu, Pending Receive = %lu\r\n",
               ulSendPktCount, ulRecvPktCount);
        }
      }
      xSysdeviceClose(hSys);
    }

    xDriverClose(hDriver);
  }

  printf(" State = 0x%08lX\r\n", lRet);
  printf("----------------------------------------------------\r\n");

  return lRet;
}

/*****************************************************************************/
/*! Function to demonstrate communication channel functionality
*   Packet Transfer and I/O Data exchange
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
static int32_t ChannelDemo()
{
  CIFXHANDLE  hDriver = NULL;
  int32_t     lRet    = xDriverOpen(&hDriver);

  printf("\r\n---------- Communication Channel demo ----------\r\n");

  if(CIFX_NO_ERROR == lRet)
  {
    /* Driver/Toolkit successfully opened */
    CIFXHANDLE hChannel = NULL;
    lRet = xChannelOpen(NULL, "cifX0", 0, &hChannel);

    if(CIFX_NO_ERROR != lRet)
    {
      printf("Error opening Channel!");

    } else
    {
      CHANNEL_INFORMATION tChannelInfo = {{0}};
      uint32_t ulCycle = 0;
      
      /* Channel successfully opened, so query basic information */
      if( CIFX_NO_ERROR != (lRet = xChannelInfo(hChannel, sizeof(CHANNEL_INFORMATION), &tChannelInfo)))
      {
        printf("Error querying system information block\r\n");
      } else
      {
        printf("Communication Channel Info:\r\n");
        printf("Device Number    : %lu\r\n", tChannelInfo.ulDeviceNumber);
        printf("Serial Number    : %lu\r\n", tChannelInfo.ulSerialNumber);
        printf("Firmware         : %s\r\n", tChannelInfo.abFWName);
        printf("FW Version       : %u.%u.%u build %u\r\n", 
                tChannelInfo.usFWMajor,
                tChannelInfo.usFWMinor,
                tChannelInfo.usFWRevision,
                tChannelInfo.usFWBuild);
        printf("FW Date          : %02u/%02u/%04u\r\n", 
                tChannelInfo.bFWMonth,
                tChannelInfo.bFWDay,
                tChannelInfo.usFWYear);

        printf("Mailbox Size     : %lu\r\n", tChannelInfo.ulMailboxSize);
      }

      CIFX_PACKET tSendPkt = {{0}};
      CIFX_PACKET tRecvPkt = {{0}};

      /* Do a basic Packet Transfer */
      if(CIFX_NO_ERROR != (lRet = xChannelPutPacket(hChannel, &tSendPkt, 10)))
      {
        printf("Error sending packet to device!\r\n");
      } else
      {
        printf("Send Packet:\r\n");
        DumpPacket(&tSendPkt);

        if(CIFX_NO_ERROR != (lRet = xChannelGetPacket(hChannel, sizeof(tRecvPkt), &tRecvPkt, 20)) )
        {
          printf("Error getting packet from device!\r\n");
        } else
        {
          printf("Received Packet:\r\n");
          DumpPacket(&tRecvPkt);
        }
      }

      /* Read and write I/O data (32Bytes). Output data will be incremented each cyle */
      uint8_t abSendData[32] = {0};
      uint8_t abRecvData[32] = {0};

      for( ulCycle = 0; ulCycle < 10; ++ulCycle)
      {
        if(CIFX_NO_ERROR != (lRet = xChannelIORead(hChannel, 0, 0, sizeof(abRecvData), abRecvData, 10)))
        {
          printf("Error reading IO Data area!\r\n");
          break;
        } else
        {
          printf("IORead Data:");
          DumpData(abRecvData, sizeof(abRecvData));

          if(CIFX_NO_ERROR != (lRet = xChannelIOWrite(hChannel, 0, 0, sizeof(abSendData), abSendData, 10)))
          {
            printf("Error writing to IO Data area!\r\n");
            break;
          } else
          {
            printf("IOWrite Data:");
            DumpData(abSendData, sizeof(abSendData));
            memset(abSendData, ulCycle + 1, sizeof(abSendData));
          }
        }
      }

      xChannelClose(hChannel);
    }

    xDriverClose(hDriver);
  }

  printf(" State = 0x%08lX\r\n", lRet);
  printf("----------------------------------------------------\r\n");

  return lRet;

}

/*****************************************************************************/
/*! Function to demonstrate control/status block functionality
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
static int32_t BlockDemo( void)
{
  CIFXHANDLE hDriver = NULL;
  int32_t    lRet    = xDriverOpen(&hDriver);
  
  printf("\n--- Read / Write Block Information ---\r\n");  

  if(CIFX_NO_ERROR == lRet)
  {
    /* Driver/Toolkit successfully opened */    
    /* Open channel */
    CIFXHANDLE hDevice = NULL;
    lRet = xChannelOpen(NULL, "cifX0", 0, &hDevice);
    if(lRet != CIFX_NO_ERROR)
    {
      printf("Error opening Channel!\r\n");
    } else
    {
      uint8_t abBuffer[4] = {0};
  
      /* Read / Write control block */
      printf("Read CONTROL Block \r\n");  
      memset( abBuffer, 0, sizeof(abBuffer));
      lRet = xChannelControlBlock( hDevice, CIFX_CMD_READ_DATA, 0, 4, &abBuffer[0]);
  
      DumpData(abBuffer, 4);
  
      printf("Write CONTROL Block \r\n");  
      lRet = xChannelControlBlock( hDevice, CIFX_CMD_WRITE_DATA, 0, 4, &abBuffer[0]);
  
      printf("Read COMMON Status Block \r\n");  
      memset( abBuffer, 0, sizeof(abBuffer));
      lRet = xChannelCommonStatusBlock( hDevice, CIFX_CMD_READ_DATA, 0, 4, &abBuffer[0]);
  
      DumpData(abBuffer, 4);
  
      printf("Write COMMON Status Block \r\n");  
      lRet = xChannelCommonStatusBlock( hDevice, CIFX_CMD_WRITE_DATA, 0, 4, &abBuffer[0]);
  
      /* this is expected to fail, as this block must not be written by Host */
      if(CIFX_NO_ERROR != lRet)
        printf("Error writing to common status block. lRet = 0x%08lx\r\n", lRet);
  
      printf("Read EXTENDED Status Block \r\n");  
      memset( abBuffer, 0, sizeof(abBuffer));
      lRet = xChannelExtendedStatusBlock( hDevice, CIFX_CMD_READ_DATA, 0, 4, &abBuffer[0]);
      DumpData(abBuffer, 4);
      
      printf("Write EXTENDED Status Block \r\n");  
      lRet = xChannelExtendedStatusBlock( hDevice, CIFX_CMD_WRITE_DATA, 0, 4, &abBuffer[0]);
  
      /* this is expected to fail, as this block must not be written by Host */
      if(CIFX_NO_ERROR != lRet)
        printf("Error writing to extended status block. lRet = 0x%08lx\r\n", lRet);
  
      xChannelClose(hDevice);
    } 
    xDriverClose(hDriver);
  }  

  return lRet;
}

/*****************************************************************************/
/*! IO Mirror demo
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
static int32_t IOMirrorDemo(void)
{
  uint8_t     abRecvData[240] = {0};
  CIFXHANDLE  hChannel        = NULL;
  CIFXHANDLE  hDriver         = NULL;
  int32_t     lRet            = CIFX_NO_ERROR;     

  printf("\n--- IO Mirror Demo ---\r\n"); 

  if ( (CIFX_NO_ERROR != (lRet = xDriverOpen(&hDriver))) ||
       (CIFX_NO_ERROR != (lRet = xChannelOpen(NULL, "cifX0", 0, &hChannel))) )
  {
    printf("Error opening Channel! (lRet = 0x%08lx)\r\n", lRet);
    
  } else
  {
    uint32_t ulCycle    = 0;
    uint32_t ulErrorCnt = 0;

    while (1)
    {
      if(CIFX_NO_ERROR != (lRet = xChannelIORead(hChannel, 0, 0, sizeof(abRecvData), abRecvData, 10)))
      {
        ulErrorCnt++;
      } else if(CIFX_NO_ERROR != (lRet = xChannelIOWrite(hChannel, 0, 0, sizeof(abRecvData), abRecvData, 10)))
      {
        ulErrorCnt++;
      }

      if (++ulCycle % 500 == 0)
      {
        printf("%lu cycles passed with %lu error(s)!\r", ulCycle, ulErrorCnt);
        fflush(stdout); 
      }
    }
  }
  
  return lRet;
}

/*****************************************************************************/
/*! Function to demonstrate communication channel functionality
*   Packet Transfer and I/O Data exchange
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
static int32_t xChannelIOPerformance (CIFXHANDLE hChannel, uint8_t bWrite, PPERFORMANCE_TEST_DATA ptPerfData, uint8_t* pabData, uint32_t ulDataLen)
{
  SYSTIME_TIMESTAMP_T tTimeStampBegin;
  SYSTIME_TIMESTAMP_T tTimeStampEnd;
  SYSTIME_TIMESTAMP_T tTimeStampDiff;
  int                 iResult;
  int32_t    lRet    = CIFX_NO_ERROR;
  uint32_t   ulDiff  = 0;


  Drv_SysTimeGetTime(&tTimeStampBegin);

  if (bWrite)
    lRet = xChannelIOWrite(hChannel, 0, 0, ulDataLen, pabData, 10);
  else
    lRet = xChannelIORead(hChannel, 0, 0, ulDataLen, pabData, 10);

  Drv_SysTimeGetTime(&tTimeStampEnd);

  Drv_SysTimeGetTimestampDifference( &tTimeStampBegin,
                                     &tTimeStampEnd,
                                     &tTimeStampDiff,
                                     &iResult);

  ulDiff = Drv_SysTimeConvertTimestampToTime(&tTimeStampDiff);

  ptPerfData->ulMin = min(ptPerfData->ulMin, ulDiff);
  ptPerfData->ulMax = max(ptPerfData->ulMax, ulDiff);
  ptPerfData->ulSum+= ulDiff;
  ptPerfData->ulCnt++;

  return lRet;
}

/*****************************************************************************/
/*! Function to demonstrate communication channel functionality
*   Packet Transfer and I/O Data exchange
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
int32_t IOPerformance(int fWriteTest)
{
  CIFXHANDLE  hDriver     = NULL;
  int32_t     lRet        = CIFX_NO_ERROR;

  printf("\n--- IO %s Performance test ---\r\n", fWriteTest==0?"Read":"Write"); 
  printf("Size\tMin\tMax\tAvg\tThp\r\n");
  printf("----------------------------------------------------\r\n");
  
  if(CIFX_NO_ERROR == (lRet = xDriverOpen(&hDriver)))
  {
    /* Driver/Toolkit successfully opened */
    CIFXHANDLE hChannel = NULL;
    lRet = xChannelOpen(NULL, "cifX0", 0, &hChannel);

    if(CIFX_NO_ERROR != lRet)
    {
      printf("Error opening Channel!");

    } else
    {
      /* Read or write I/O data (32Bytes). Output data will be incremented each cyle */
      uint8_t pabData[240];
      int     iIOSize        = 0;

      for (iIOSize = 20; iIOSize<=240; iIOSize+=20)
      {
        uint32_t              ulCycle    = 0;
        PERFORMANCE_TEST_DATA tPerfData  = {0, ~0, 0, 0};

        for( ulCycle = 0; ulCycle < 100; ++ulCycle)
        {
          if( (CIFX_NO_ERROR != (lRet = xChannelIOPerformance(hChannel, fWriteTest, &tPerfData, pabData, iIOSize))) &&
              (CIFX_DEV_NO_COM_FLAG != lRet ) )
          {
            printf("Error reading IO Data area!\r\n");
            break;
          } else
          {
            memset(pabData, ulCycle + 1, iIOSize);
          }
        }
        if( (CIFX_NO_ERROR == lRet) || (CIFX_DEV_NO_COM_FLAG == lRet ) )
        {
          printf("%d\t%ldus\t%ldus\t%ldus\t%lld KBit/s\r\n",iIOSize,
                                                            tPerfData.ulMin/1000,
                                                            tPerfData.ulMax/1000,
                                                            (tPerfData.ulSum/tPerfData.ulCnt)/1000,
                                                            (uint64_t)iIOSize*8000000/(tPerfData.ulSum/tPerfData.ulCnt));
        }
      }
      xChannelClose(hChannel);
    }

    xDriverClose(hDriver);
  }

  printf("----------------------------------------------------\r\n");
  return lRet;
}

/*****************************************************************************/
/*! Assert chip select
*   \param pvOSDependent OS Dependent parameter to identify card             */
/*****************************************************************************/
void OS_SpiAssert(void* pvOSDependent)
{
  RCX_DEVICEINSTANCE_T* ptrcXInst = (RCX_DEVICEINSTANCE_T*)pvOSDependent;
  Drv_SpiOpen(ptrcXInst->hSerDPM, RX_INFINITE);
  Drv_SpiSlaveSelect(ptrcXInst->hSerDPM);
}

/*****************************************************************************/
/*! Deassert chip select
*   \param pvOSDependent OS Dependent parameter to identify card             */
/*****************************************************************************/
void OS_SpiDeassert(void* pvOSDependent)
{
  RCX_DEVICEINSTANCE_T* ptrcXInst = (RCX_DEVICEINSTANCE_T*)pvOSDependent;
  Drv_SpiSlaveDeselect(ptrcXInst->hSerDPM);
  Drv_SpiClose(ptrcXInst->hSerDPM);
}

/*****************************************************************************/
/*! Transfer byte stream via SPI
*   \param pvOSDependent OS Dependent parameter to identify card
*   \param pbSend        Send buffer (NULL for polling)
*   \param pbRecv        Receive buffer (NULL if discard)
*   \param ulLen         Length of SPI transfer                              */
/*****************************************************************************/
void OS_SpiTransfer(void* pvOSDependent, uint8_t* pbSend, uint8_t* pbRecv, uint32_t ulLen)
{
  RCX_DEVICEINSTANCE_T* ptrcXInst = (RCX_DEVICEINSTANCE_T*)pvOSDependent;
  Drv_SpiSendReceive(ptrcXInst->hSerDPM, ulLen, pbSend, pbRecv);
}

/*****************************************************************************/
/*! Toolkit initialization function
*   \param argc
*   \param argv                                                              */
/*****************************************************************************/
void ToolkitSampleTask(void* pvParam)
{
  TLR_HANDLE hReadyLed   = NULL;
  TLR_HANDLE hRunLed     = NULL;
  int32_t    lTkRet      = CIFX_NO_ERROR;

  /* Signal task initialized */
  rX_SysSetTaskInitialized(NULL);    
  
  /* Init Sys LED */
  Drv_LedIdentifyLed("READY",0, &hReadyLed);
  Drv_LedIdentifyLed("RUN",0, &hRunLed);
  Drv_LedInitializeLed(hReadyLed);
  Drv_LedInitializeLed(hRunLed);
  Drv_LedSetLed(hRunLed);

  /* Redirect printf output to uart */
  UartStdIOHookInit("REALUART", "UART");
      
  /* First of all initialize toolkit */
  lTkRet = cifXTKitInit();

  if(CIFX_NO_ERROR == lTkRet)
  {
    RCX_DEVICEINSTANCE_T* ptInstance = (RCX_DEVICEINSTANCE_T*)OS_Memalloc(sizeof(*ptInstance));
    UINT32                ulSerDPMType;

    /* Set trace level of toolkit */
    g_ulTraceLevel = TRACE_LEVEL_ERROR   |
                     TRACE_LEVEL_WARNING |
                     TRACE_LEVEL_INFO    |
                     TRACE_LEVEL_DEBUG;

    OS_Memset(ptInstance, 0, sizeof(*ptInstance));
    ptInstance->tDevice.pvOSDependent = ptInstance;
    ptInstance->tDevice.ulDPMSize     = 0x10000;
    OS_Strncpy(ptInstance->tDevice.szName, "cifX0", sizeof(ptInstance->tDevice.szName));

    /* Init SPI */
    if ( (RX_OK != Drv_SpiIdentifySpi("SERDPM", 0, &ptInstance->hSerDPM)) ||
         (RX_OK != Drv_SpiInitializeSpi(ptInstance->hSerDPM)) )
    {
      printf("SPI initialization failed!)\r\n");
    } else if (SERDPM_UNKNOWN == (ulSerDPMType = SerialDPM_Init(&ptInstance->tDevice)))
    {
      printf("Serial DPM protocol could not be recognized!)\r\n");
    } else
    {
      char* szSerDPM[] = {"", "NETX10", "NETX50", "NETX51", "NETX100"};

      printf("Serial DPM protocol detected: %s\r\n", szSerDPM[ulSerDPMType]);

      /* Add the device to the toolkits handled device list */
      lTkRet = cifXTKitAddDevice(&ptInstance->tDevice);
  
      /* If it succeeded do device tests */    
      if(CIFX_NO_ERROR != lTkRet)
      {
        /* Uninitialize Toolkit, this will remove all handled boards from the toolkit and 
           deallocate the device instance */
        cifXTKitDeinit();
      } else
      {
        /* Create Polling thread */
        rX_SysCreateTask("PollingTask",
                         cifXPollingThread,
                         (void*)500,
                         NULL,
                         100,
                         RX_TASK_AUTO_START,
                         0,
                         TSK_PRIO_37,
                         TSK_TOK_37,
                         0,
                         NULL);
                             
        /* Signale toolkit init success */
        Drv_LedSetLed(hReadyLed); 
      
        /* Toolkit demonstration */
        EnumBoardDemo();
        SysdeviceDemo();
        ChannelDemo();
        BlockDemo();
        IOPerformance(0);
        IOPerformance(1);
        IOMirrorDemo();      
      }
    }
  }

  for (;;)
    rX_SysSleepTask(1000);
}


